//==================================================================
// Licensed Materials - Property of IBM
// (c) Copyright IBM Corp. 2003.  All rights reserved.
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM
// Corp.
//
// DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
// sample code created by IBM Corporation.  This sample code is
// not part of any standard or IBM product and is provided to you
// solely for the purpose of assisting you in the development of
// your applications.  The code is provided "AS IS", without
// warranty of any kind.  IBM shall not be liable for any damages
// arising out of your use of the sample code, even if they have
// been advised of the possibility of such damages.
//==================================================================

var ODCCommonNLSService = new NLSService();


/**
 * @class public NLSService
 * This class provides all the necessary apis for currency, number, date and time NLE
 * @constructor NLSService
 *   The constructor takes two parameters and defaults the locale is "en_US" and the decimal place
 *	 is 2.
 * @param String locale
 *	Which locale is used to NLE the data.
 * @param int decimalPlace
 *  which is used to format the currency and number. This number will overwrite the default decimal
 *  places which is used defaultly in that locale.
**/

function NLSService(locale, decimalPlace)
{

	this.locale = (locale!=null&&typeof(locale)!='undefined')?expandToFullLocale(locale):"en_US";
	this.decimalPlace =(decimalPlace<0)?0:decimalPlace;

/**
* @method public expandToFullLocale
*   Method to get the full qualified locale based on the short one
* @param String loc
*   The short name of that local
* @return String fullLocale
*   The full qualified locale.
**/
	function expandToFullLocale(loc)
	{
		var fullLocale = loc;

		switch (loc)
		{
			case "ar": fullLocale = "ar_SA"; break; // assumes Saudi Arabia for Arabic. OK?
			case "cs": fullLocale = "cs_CZ"; break;
			case "da": fullLocale = "da_DK"; break;
			case "de": fullLocale = "de_DE"; break;
			case "el": fullLocale = "el_GR"; break;
			case "en": fullLocale = "en_US"; break;
			case "es": fullLocale = "es_ES"; break; // OK?
			case "fi": fullLocale = "fi_FI"; break;
			case "fr": fullLocale = "fr_FR"; break; // OK?
			case "iw":
			case "he": fullLocale = "he_IL"; break;
			case "hu": fullLocale = "hu_HU"; break;
			case "it": fullLocale = "it_IT"; break;
			case "ja": fullLocale = "ja_JP"; break;
			case "ko": fullLocale = "ko_KR"; break;
			case "no": fullLocale = "no_NO"; break;
			case "nl": fullLocale = "nl_NL"; break;
			case "pl": fullLocale = "pl_PL"; break;
			case "pt": fullLocale = "pt_PT"; break;
			case "ru": fullLocale = "ru_RU"; break;
			case "sv": fullLocale = "sv_SE"; break;
			case "tr": fullLocale = "tr_TR"; break;
			case "zh": fullLocale = "zh_CN"; break;
			default: break;
		}
		return fullLocale;
	}
/**
* @method public setLocale
*   Method to set the locale
* @param String value
*   The name of that local
**/

	this.setLocale = function(value)
	{
		this.locale = expandToFullLocale(value);


	}
/**
* @method public getLocale
*   Method to get the locale
* @return String locale
*   The name of that local
**/

	this.getLocale = function()
	{
		return this.locale;
	}
/**
* @method public setDecimalPlace
*   Method to set the decimal place
* @param int value
*   The decimal place which user use to overwrite the default one.
**/
	this.setDecimalPlace = function(value)
	{
		this.decimalPlace = value;
	}
/**
* @method public constructDateFormat
*   Method to construct the date format variable in property file.
* @param String format
*   the date format user picked like dd/mm/yy
* @return String date format
*	Returns the date format like $$DD$/$$MM$/$$YY$
**/

	this.constructDateFormat = function(format)
	{
		var dateFrmt;
		if(format!=null&&format!=""){
			var formatArray=format.split(/\W+/);

			for(var i=0; i<formatArray.length; i++){
				format = format.replace(formatArray[i], "$$"+ formatArray[i].toUpperCase()+"$");

			}
			dateFrmt = format;

		}
		return dateFrmt;
	}


/**
* @method public constructTimeFormat
*   Method to construct the time format variable in property file.
* @param String format
*   the time format user picked like hh:mm:ss
* @return String time format
*	Returns the time format like $$HH$:$$MM$:$$SS$
**/

	this.constructTimeFormat = function(format)
	{
		var timeFrmt;
		if(format!=null&&format!=""){
			var formatArray=format.split(/\W+/);

			for(var i=0; i<formatArray.length; i++){
				format = format.replace(formatArray[i], "$$"+ formatArray[i].toUpperCase()+"$");

			}
			timeFrmt = format;

		}


		return timeFrmt;
	}



/**
* @method public getFormattedCurrency
*   Method to get formatted currency based on the input currency object
* @param Object value
*   The currency to be handled which is currency object
* @return String time format
*	Returns the formatted currency data like "1,234.23".
**/


	this.getFormattedCurrency = function(value)
	{

		var number=/[0-9]/;


		var valueAmount = value.currencyAmount;

		if((valueAmount.indexOf("-")!=-1)||(valueAmount.indexOf("(")!=-1))
		{//negtive data

			var pos = valueAmount.search(number);

			valueAmount = valueAmount.substring(pos);


		}


		var absValue = new currency(valueAmount+"_"+value.currencyDesc)
		var val = this.formatCurrency(absValue.currencyAmount, absValue.currencyDesc, this.locale);

		return val;

	}


/**
* @method public getThousandSeperator
*   Method to get monetory thousand seperator for the current locale
* @return String thousand seperator
*	Returns the thousand seperator of the current locale
**/
	this.getThousandSeperator = function()
	{
		  if(this.locale != null && LocaleFormats[this.locale]!=null&&LocaleFormats[this.locale]!='undefined')
		  {
			 return LocaleFormats[this.locale][0];
		  }
		  else
		  {
			return LocaleFormats["default"][0];
		  }

	}
/**
* @method public getDecimalSeperator
*   Method to get monetory decimal seperator for the current locale
* @return String decimal seperator
*	Returns the decimal seperator of the current locale
**/
	this.getDecimalSeperator = function()
	{

		  if(this.locale != null && LocaleFormats[this.locale]!=null&&LocaleFormats[this.locale]!='undefined')
		  {
			 return LocaleFormats[this.locale][2];
		  }
		  else
		  {
			return LocaleFormats["default"][2];
		  }

	}


/**
* @method public getCurrencyPrefix
*   Method to get the prefix symbol for the current locale based on the input currency object.
* @param Object value
*	 The currency to be handled which is currency object
* @param String type
*	The symbol type the user want. support html entities or unicode
* @return
* 	Returns the prefix of currency data like "$".or "-$" for negtive result.
**/
	this.getCurrencyPrefix = function(value,type)
	{
		var prefixPos,prefixNeg;
		if(value.currencyDesc != null && CurrencyFormats[value.currencyDesc]!=null&&CurrencyFormats[value.currencyDesc]!='undefined')
		{
			if(value.currencyDesc == "CAD"){
				if(this.locale=="fr_CA"){
					prefixPos = (type == 'html') ? CurrencyFormats[value.currencyDesc][1][0] : CurrencyFormats[value.currencyDesc][1][6];
					prefixNeg = (type == 'html') ? CurrencyFormats[value.currencyDesc][1][1] : CurrencyFormats[value.currencyDesc][1][7];

				}else{

					prefixPos = (type == 'html') ? CurrencyFormats[value.currencyDesc][0][0] : CurrencyFormats[value.currencyDesc][0][6];
					prefixNeg = (type == 'html') ? CurrencyFormats[value.currencyDesc][0][1] : CurrencyFormats[value.currencyDesc][0][7];
				}

			}else{

				prefixPos = (type == 'html') ? CurrencyFormats[value.currencyDesc][0] : CurrencyFormats[value.currencyDesc][6];
				prefixNeg = (type == 'html') ? CurrencyFormats[value.currencyDesc][1] : CurrencyFormats[value.currencyDesc][7];
			}

		}else{
			prefixPos = (type == 'html') ? CurrencyFormats['default'][0] : CurrencyFormats['default'][6];
			prefixNeg = (type == 'html') ? CurrencyFormats['default'][1] : CurrencyFormats['default'][7];

		}
		var valueAmount = value.currencyAmount;

		if(prefixNeg!=null &&((valueAmount.indexOf("-")!=-1)||(valueAmount.indexOf("(")!=-1)))
		{//negtive data
			return prefixNeg;
		}else if(prefixPos!=null){
			return prefixPos;
		}else{
			return "";
		}
	}
/**
* @method public getCurrencySuffix
*   Method to get the suffix symbol for the current locale based on the input currency object.
* @param Object value
*	 The currency to be handled which is currency object
* @param String type
*	The symbol type the user want. support html entities or unicode
* @return
* 	Returns the suffix of currency data like "LUF".
**/


	this.getCurrencySuffix = function(value, type)
	{
		var suffixPos, suffixNeg;
		if(value.currencyDesc != null && CurrencyFormats[value.currencyDesc]!=null&&CurrencyFormats[value.currencyDesc]!='undefined')
		{
			if(value.currencyDesc == "CAD"){
				if(this.locale=="fr_CA"){
					suffixPos = (type == 'html') ? CurrencyFormats[value.currencyDesc][1][2]:CurrencyFormats[value.currencyDesc][1][8];
					suffixNeg = (type == 'html') ? CurrencyFormats[value.currencyDesc][1][3]:CurrencyFormats[value.currencyDesc][1][9];
				}else{

					suffixPos = (type == 'html') ? CurrencyFormats[value.currencyDesc][0][2]:CurrencyFormats[value.currencyDesc][0][8];
					suffixNeg = (type == 'html') ? CurrencyFormats[value.currencyDesc][0][3]:CurrencyFormats[value.currencyDesc][0][9];

				}


			}else{

				suffixPos = (type == 'html') ? CurrencyFormats[value.currencyDesc][2]:CurrencyFormats[value.currencyDesc][8];
				suffixNeg = (type == 'html') ? CurrencyFormats[value.currencyDesc][3]:CurrencyFormats[value.currencyDesc][9];
			}
		}else{

			suffixPos = (type == 'html') ? CurrencyFormats['default'][2]:CurrencyFormats['default'][8];
			suffixNeg = (type == 'html') ? CurrencyFormats['default'][3]:CurrencyFormats['default'][9];

		}

		var valueAmount = value.currencyAmount;
		if(suffixNeg !=null&&((valueAmount.indexOf("-")!=-1)||(valueAmount.indexOf("(")!=-1)))
		{//negtive data
			return suffixNeg;
		}else if(suffixPos !=null){
			return suffixPos;
		}else{
			return "";
		}


	}

/**
* @method public printCurrency
*   Method to print out the formatted currency data
* @param Object value
*	 The currency to be handled which is currency object
* @return String
* 	The formatted currency string of currency data like "1,234.56"
**/

	this.printCurrency = function(value)
	{
		return this.getFormattedCurrency(value);
	}

/**
* @method public isValidCurrency
*   Method to detect if it a valid currency
* @param Object value
*	 The currency to be handled which is currency object
* @return boolean
* 	Returns true if it is , otherwise false
**/

	this.isValidCurrency = function(value)
	{
		return isValidCur(value.currencyAmount, value.currencyDesc, this.locale);

	}


/**
* @method public getUnformattedCurrency
*   Method to get the unformatted currency string
* @param Object value
*	 The formatted currency object like: $1,234.56_USD
* @param boolean prefix
*	if true, means the prefix need to be removed.
* @param boolean suffix
*	if true, means the suffix need to be removed.
* @return String
* 	Returns unformatted currency object string: 1234.56_USD
**/

	this.getUnformattedCurrency = function(value, prefix, surfix)
	{
		var valueAmount = value.currencyAmount;
		var valuedes = value.currencyDesc;

		if(prefix)
			valueAmount = removePrefix(valueAmount);
		if(surfix)
			valueAmount = removeSuffix(valueAmount);
		var valueStr = valueAmount +"_"+valuedes;

		value = new currency(valueStr);

		var val = getUnformattedCur(value.currencyAmount,  this.locale);
		return (val+"_"+value.currencyDesc);
	}

/**
* @method public isValidDate
*   Method to detect if it a valid date
* @param Object value
*	 The date to be handled which is date object
* @return boolean
* 	Returns true if date is valid , otherwise false
**/

	this.isValidDate = function(date)
	{

		return validDate(date.year, date.month, date.day);

	}

/**
* @method public getUnformattedDate
*   Method to get the unformatted date string
* @param String value
*	 The formatted date like: 2002/3/15
* @return String
* 	Returns unformatted date object string like: 3/15/2002
**/

	this.getUnformattedDate = function(value)
	{
         //alert("getUnformattedDate()");
			value = value.toString();
			var dateInfoArray=value.split(/\D+/);

			var i=0;
			var tokenName;
			var valueArray = new Array();

			var tokenStart = dateFormat.indexOf( "$$" );
			var tokenEnd;

			while( tokenStart != -1 )
			{
			   tokenEnd = dateFormat.indexOf( "$", tokenStart+2 );
			   tokenName = dateFormat.substring(tokenStart +2, tokenEnd);
			   valueArray[tokenName] = dateInfoArray[i++];

			   nCursor = tokenEnd + 1;
			   tokenStart = dateFormat.indexOf( "$$", nCursor);


			}

         if(valueArray["YYYY"]!="" && valueArray["YYYY"]!=null && valueArray["MM"]!="" && valueArray["MM"]!=null && valueArray["DD"]!="" && valueArray["DD"]!=null)
         {
            var aDate = new Date(valueArray["YYYY"],  valueArray["MM"]-1, valueArray["DD"]) ;
            return value = aDate.getTime();
         }
         else
         {
            
			   var Msg = NlsFormatMsg(invalid_date_format, null);

			   Log.error("NLSService.getUnformattedDate", Msg);

			   throw new NLSServiceError(Msg);
            
         }

         /*
 			return  ((valueArray["YYYY"]!=""&&valueArray["YYYY"]!=null) ? valueArray["YYYY"]+"/":"")
                   +((valueArray["MM"]!=""&&valueArray["MM"]!=null) ? valueArray["MM"]+"/":"")
                   +((valueArray["DD"]!=""&&valueArray["DD"]!=null) ? valueArray["DD"]:"");
                   */

	}


/**
* @method public getDateMask
*   Method to get the date format array for per locale.Lily requested.
* @return Array
* 	Returns an array which has all the date format for that locale
**/

	this.getDateMask = function()
	{

		var curLocale;

		if(this.locale != null && DateFormats[this.locale]!=null&& DateFormats[this.locale]!='undefined')
		{
		  curLocale = this.locale;
		}
		else
		{
		  curLocale = "default";
		}


		var dtFmtArray = DateFormats[curLocale];

		var dtFmtLength = dtFmtArray.length;
		var maskArray = new Array(dtFmtLength);
		for(var i=0; i<dtFmtLength; i++){

			var perFormat = dtFmtArray[i][0];
			perFormat = perFormat.replace(/\$+/g, "");
			maskArray[i] = perFormat;

		}
		return maskArray;

	}

/**
* @method public getFormattedUsrInputDate
*   Method to get the formatted date string from the user input. Lily requested.
* @param String format
*	The date format from spreadsheet, expect: mm/dd/yy
* @param String/int
*	This parameter can be two cases,
*		The date string which should has seperator among mm,dd,yy like mm.dd.yy
* 		or a big integer which is milliseconds counted from 1/1/1970.
* @return String
* 	Returns formatted date object string: 3/15/2002
* @exception NLSServiceError
*	if the passed parameter format is null
*	or the passes date is not a valid date.
* 	or error when format the date based on the format style.
**/

	this.getFormattedUsrInputDate = function(format, value)
	{

		if(format==null){
			var Msg = NlsFormatMsg(invalid_date_format, null);

			Log.error("NLSService.getFormattedUsrInputDate", Msg);

			throw new NLSServiceError(Msg);

		}

		format = new String(format).toUpperCase();

		//var formatArr = format.split(/\W+/);
		var formatArr = format.split(/[^Y*M+D+]/);
		//var formatArr = format.split(/M+/);

		var dateValues = new Array();
		var yy=mm=dd=0;

		if(isNaN(value)){
			var dateInfoArray=value.split(/\D+/);

			for(var i=0; i<dateInfoArray.length; i++)
			{

				dateValues[formatArr[i]] = dateInfoArray[i];


			}
		}else{
			var dateObj = new Date(Number(value));
			yy = dateObj.getUTCFullYear();

			mm = dateObj.getUTCMonth()+1;

			dd = dateObj.getUTCDate();


		}

		var dateFrmt = format;
		var index = format.indexOf("YY");
		if(index!=-1)
		{
			if(format.indexOf("YYYY")==-1){
				if(index!=-1){
					dateFrmt = dateFrmt.substring(0,index)+"$$YY$"+dateFrmt.substring(index+2);
					if(!isNaN(value))
						dateValues["YY"] = String(yy);

				}

			}else{
				dateFrmt = dateFrmt.substring(0,index)+"$$YYYY$"+dateFrmt.substring(index+4);
				if(!isNaN(value))
					dateValues["YYYY"] = String(yy);

			}


		}

		index = dateFrmt.indexOf("MM");
		if(index==-1)
		{
			// m instead of mm
			index = dateFrmt.indexOf("M");
			if(index!=-1){
				dateFrmt = dateFrmt.substring(0,index)+"$$M$"+dateFrmt.substring(index+1);
				if(!isNaN(value))
					dateValues["M"] = String(mm);
			}


		}else{
			dateFrmt = dateFrmt.substring(0,index)+"$$MM$"+dateFrmt.substring(index+2);
			if(!isNaN(value))
				dateValues["MM"] = String(mm);
		}

		index = dateFrmt.indexOf("DD");
		if(index==-1)
		{
			index = dateFrmt.indexOf("D");
			if(index!=-1){
				dateFrmt = dateFrmt.substring(0,index)+"$$D$"+dateFrmt.substring(index+1);
				if(!isNaN(value))
					dateValues["D"] = String(dd);
			}



		}else{

			dateFrmt = dateFrmt.substring(0,index)+"$$DD$"+dateFrmt.substring(index+2);
			if(!isNaN(value))
				dateValues["DD"] = String(dd);


		}

		 	var index_yy = dateFrmt.indexOf("YYYY");
		 	var index_mm = dateFrmt.indexOf("MM");
		 	var index_dd = dateFrmt.indexOf("DD");

			var temp_date = new Date();
			var temp_year = temp_date.getFullYear();
			var temp_month, temp_day;
			//based on Microsoft excel, we hardcode to use 29 as cutoff. so if user enter 29, then display 2029.
			//if user enter 30, then return 1930
			var NUM_CENTYEAR = 29;

			if(index_yy!=-1&&dateValues["YYYY"]!=null&&dateValues["YYYY"]!="")
			{


				if(Number(dateValues["YYYY"])<100){

					dateValues["YYYY"] = Number(dateValues["YYYY"]) + (dateValues["YYYY"] <= NUM_CENTYEAR ? 2000 : 1900);

				}

				temp_year = dateValues["YYYY"];

			}

			if(index_yy==-1&&dateValues["YY"]!=null&&dateValues["YY"]!="")
			{
				if(Number(dateValues["YY"])>100){


					dateValues["YY"] = dateValues["YY"].substring(2);

				}

				temp_year = Number(dateValues["YY"]) + (dateValues["YY"] <= NUM_CENTYEAR ? 2000 : 1900);


			}

			if(index_mm!=-1&&dateValues["MM"]!=null&&dateValues["MM"]!="")
			{

				if(String(dateValues["MM"]).indexOf("0") !=0){

					dateValues["MM"] = ((dateValues["MM"]  < 10) ? "0":"") + dateValues["MM"];
				}

				temp_month =dateValues["MM"];



			}else{
				if(dateValues["M"]!=null&&dateValues["M"].length>1&&dateValues["M"].indexOf("0") ==0){
					dateValues["M"] = dateValues["M"].substring(1);

				}
				temp_month = dateValues["M"];
			}
			if(index_dd!=-1&&dateValues["DD"]!=null&&dateValues["DD"]!="")
			{

				if(String(dateValues["DD"]).indexOf("0") !=0){

					dateValues["DD"] = ((dateValues["DD"]  < 10) ? "0":"") + dateValues["DD"];
				}

				temp_day = dateValues["DD"];

			}else{
				if(dateValues["D"]!=null&&dateValues["D"].length>1&&dateValues["D"].indexOf("0") ==0){
					dateValues["D"] = dateValues["D"].substring(1);

				}


				temp_day = dateValues["D"];
			}


			temp_month = (temp_month!=null&&temp_month!='undefined')?temp_month:"1";
			temp_day = (temp_day!=null&&temp_day!='undefined')?temp_day:"1";



			if(!validDate(new String(temp_year), temp_month, temp_day)){
				var args = new Array;
				args[0] = value;
				args[1] = format;
				var Msg = NlsFormatMsg(invalid_date, args);
				Log.error("NLSService.getFormattedUsrInputDate", Msg);

				throw new NLSServiceError(Msg);
			}


			var temp = replaceTokens(dateValues, dateFrmt);
			if(temp!="exception")
				return temp;
			else{
				var args = new Array;
				args[0] = value;
				args[1] = format;
				var Msg = NlsFormatMsg(invalid_date, args);
				Log.error("NLSService.getFormattedUsrInputDate", Msg);

				throw new NLSServiceError(Msg);
			}

			//return replaceTokens(dateValues, dateFrmt);





	}

/**
* @method public getFormattedDate
*   Method to get the formatted date string.
* @param Object value
*	The date object like: new date('2002/3/15');
* @return String
* 	Returns formatted date object string: 3/15/2002
* @exception NLSServiceError
*	if the passes date is not a valid date.
* 	or error when format the date based on the format style.
**/

	this.getFormattedDate = function(value)
	{

			var dateValues = new Object();

			if(dateFormat==null){
				var Msg = NlsFormatMsg(invalid_date_format, null);
				Log.error("NLSService.getFormattedDate", Msg);

				throw new NLSServiceError(Msg);
			}

            //var yy = value.year;
            //var mm = value.month;
	        //var dd = value.day;


         var yy = value.getFullYear();
         var mm = value.getMonth()+ 1;
	      var dd = value.getDate();


            var index_yy = dateFormat.indexOf("YYYY");
            if(index_yy==-1){
				var tempIndex = dateFormat.indexOf("YY");
				if(tempIndex!=-1)
					dateValues["YY"] = yy;
			}else{
				dateValues["YYYY"] = yy;
			}

			var index_mm = dateFormat.indexOf("MM");
			if(index_mm==-1){
				var tempIndex = dateFormat.indexOf("M");
				if(tempIndex!=-1)
					dateValues["M"] = mm;
			}else{
				dateValues["MM"] = mm;
			}

			var index_dd = dateFormat.indexOf("DD");
			if(index_dd==-1){
				var tempIndex = dateFormat.indexOf("D");
				if(tempIndex!=-1)
					dateValues["D"] = dd;
			}else
				dateValues["DD"] = dd;

			var temp_date = new Date();
			var temp_year = temp_date.getFullYear();
			var NUM_CENTYEAR = new String(temp_year).substring(2);

			if(index_yy!=-1&&dateValues["YYYY"]!=null&&dateValues["YYYY"]!="")
			{


				if(Number(dateValues["YYYY"])<100){

					dateValues["YYYY"] = Number(dateValues["YYYY"]) + (dateValues["YYYY"] < NUM_CENTYEAR ? 2000 : 1900);

				}

			}

			if(index_mm!=-1&&dateValues["MM"]!=null&&dateValues["MM"]!="")
			{
				dateValues["MM"] = ((dateValues["MM"]  < 10) ? "0":"") + dateValues["MM"];


			}

			if(index_dd!=-1&&dateValues["DD"]!=null&&dateValues["DD"]!="")
			{
				dateValues["DD"] = ((dateValues["DD"]  < 10) ? "0":"") + dateValues["DD"];
			}

			var temp = replaceTokens(dateValues, dateFormat);
			if(temp!="exception")
				return temp;
			else{
				var args = new Array;
				args[0] = value;
				args[1] = dateFormat;
				var Msg = NlsFormatMsg(invalid_date, args);
				Log.error("NLSService.getFormattedUsrInputDate", Msg);

				throw new NLSServiceError(Msg);
			}



	}

/**
* @method public printDate
*   Method to print the formatted date object
* @param Object value
*	The date object like: new date('2002/3/15');
* @return String
* 	Returns formatted date object string: 3/15/2002
* @exception NLSServiceError
*	if the passes date is not a valid date.
* 	or error when format the date based on the format style.
**/

	this.printDate = function(value)
	{

		return this.getFormattedDate(value);
	}



/**
* @method public isValidTime
*   Method to detect if it a valid time
* @param Object time
*	The time object which is to be handled
* @return boolean
* 	Returns true if the time is valid, otherwise false
**/
	this.isValidTime = function(time)
	{

		return validTime(time.hour, time.minute, time.second);

	}
/**
* @method public getUnformattedTime
*   Method to get the unformatted time string
* @param String value
*	The formatted time like: 23hour25min44second
* @return String
* 	Returns unformatted time object string: 23:25:44
**/

	this.getUnformattedTime = function(value)
	{

		if(value!=null&&value!=""){

			value = value.toString();
			var timeInfoArray=value.split(/\D+/);

			var ampm = "";
			var needAMPM = "false";
			var index = timeFormat.indexOf("HH");

			if( AM!=""&&AM!=null&&PM!=""&&PM!=null&&index==-1){
				needAMPM = "true";
				ampm = value.split(" ")[1];

			}


			var i=0;
			var tokenName;
			var valueArray = new Array();

			var tokenStart = timeFormat.indexOf( "$$" );
			var tokenEnd;

			while( tokenStart != -1 )
			{
			   tokenEnd = timeFormat.indexOf( "$", tokenStart+2 );
			   tokenName = timeFormat.substring(tokenStart +2, tokenEnd);
			   valueArray[tokenName] = timeInfoArray[i++];

			   nCursor = tokenEnd + 1;
			   tokenStart = timeFormat.indexOf( "$$", nCursor);


			}
			var hours,minutes,seconds;

			if(index==-1)
				hours = valueArray["H"];
			else
				hours = valueArray["HH"];

			minutes = valueArray["MM"];

			seconds = valueArray["SS"];

			if( needAMPM=="true")
			{
				if(ampm==PM && hours<12){

					hours = Number(hours)+12;
				}

			}

			 hours =  (hours.substring(0,1)=="0") ? (hours.substring(1)):hours;
			 minutes = (minutes.substring(0,1)=="0") ? (minutes.substring(1)):minutes;
			 seconds = (seconds.substring(0,1)=="0") ? (seconds.substring(1)):seconds;


			//return  ((valueArray["HH"]!=""&&valueArray["HH"]!=null) ? valueArray["HH"]+":":"")
			//	   +((valueArray["MM"]!=""&&valueArray["MM"]!=null) ? valueArray["MM"]+":":"")
			//	   +((valueArray["SS"]!=""&&valueArray["SS"]!=null) ? valueArray["SS"]:"");
			return hours+":"+minutes+":"+seconds;
		}else
			return "";

	}
/**
* @method public getTimeMask
*   Method to get the time format array for per locale. Lily requested.
* @return Array
* 	Returns an array which has all the time format for that locale
**/
	this.getTimeMask = function()
	{
		var curLocale;

		if(this.locale != null && DateFormats[this.locale]!=null&& DateFormats[this.locale]!='undefined')
		{
		  curLocale = this.locale;
		}
		else
		{
		  curLocale = "default";
		}

		var tmFmtArray = TimeFormats[curLocale];
		var tmFmtLength = tmFmtArray.length;
		var maskArray = new Array(tmFmtLength);
		for(var i=0; i<tmFmtLength; i++){

			var perFormat = tmFmtArray[i][0];
			perFormat = perFormat.replace(/\$+/g, "");
			maskArray[i] = perFormat;

		}

		return maskArray;

	}



/**
* @method public getTimeAmpm
*   Method to get the time am pm format for that locale. Lily requested.
* @return String
* 	Returns a string array like "AM/PM" for that locale if it has am/pm information
**/
	this.getTimeAmpm = function()
	{
		var curLocale;

		if(this.locale != null && DateFormats[this.locale]!=null&& DateFormats[this.locale]!='undefined')
		{
		  curLocale = this.locale;
		}
		else
		{
		  curLocale = "default";
		}

		var tmFmtArray = TimeFormats[curLocale];
		var tmFmtLength = tmFmtArray.length;
		var ampmArray = new Array();
		for(var i=0; i<tmFmtLength; i++){

			var am = tmFmtArray[i][2];
			var pm = tmFmtArray[i][3];
			if(am!='undefined'&&am!=null&&pm!='undefined'&&pm!=null)
				ampmArray[ampmArray.length] = am+"_"+pm;
			else
				ampmArray[ampmArray.length] = "";

		}

		return ampmArray;

	}



/**
* @method public getTimeAmpm
*   Method to get the time am pm formats for all locales. Lily requested.
* @return Array
* 	Returns an Array which has all the ampm formats info for all locales.
**/
	this.getAllTimeAmpms = function()
	{
			var tempArray = new Array();
			var tmFmtArray = new Array();
			var am,pm;

				for ( var a in TimeFormats )
				{


					tmFmtArray = TimeFormats[a];


					//since all the locales have the same am/pm information. so just take data from the first array.
					am = tmFmtArray[0][2];
					pm = tmFmtArray[0][3];
					if(am!='undefined'&&am!=null&&pm!='undefined'&&pm!=null)
						tempArray[am+"_"+pm] = 1;

				}
				var ampmArray = new Array();

				for(var a in tempArray)
					ampmArray[ampmArray.length] = a;

				return ampmArray;



	}



/**
* @method public getTimeAmpmPos
*   Method to get the time am pm position for that locale. Lily requested.
* @return Array
* 	Returns an Array where the string element "PRE" means am/pm info is in the front for that locale
*	"POST" means am.pm is after the time.
**/
	this.getTimeAmpmPos = function()
	{
		var curLocale;

		if(this.locale != null && DateFormats[this.locale]!=null&& DateFormats[this.locale]!='undefined')
		{
		  curLocale = this.locale;
		}
		else
		{
		  curLocale = "default";
		}

		var tmFmtArray = TimeFormats[curLocale];
		var tmFmtLength = tmFmtArray.length;
		var ampmPosArray = new Array();
		for(var i=0; i<tmFmtLength; i++){

			var ampmPos = tmFmtArray[i][1];

			if(ampmPos!='undefined'&&ampmPos!=null)
				ampmPosArray[ampmPosArray.length] = ampmPos;
			else
				ampmPosArray[ampmPosArray.length] = "";

		}

		return ampmPosArray;
	}



/**
* @method public getFormattedUsrInputTime
*   Method to get the formatted time string from the user input. Lily requested.
* @param String format
*	The format user pick up from pull down menu (h:m:s).
* 	h means need display am, pm. hh means no am, pm.
* @param String
*	The time string which should has seperator among hh,mm,ss, (2:12:14)
* @return String
* 	Returns formatted time object string: 2unicode12unicode13unicode [am/pm]
* @exception NLSServiceError
*	if the passed parameter format is null
*	or the passes time is not a valid time.
* 	or error when format the time based on the format style.
**/
	this.getFormattedUsrInputTime = function(format, value)
	{



		if(format==null){
			var Msg = NlsFormatMsg(invalid_time_format, null);
			Log.error("NLSService.getFormattedUsrInputTime", Msg);


			throw new NLSServiceError(Msg);

		}

		var index = format.indexOf("_");
		var timemask, ampm, ampmPos;

		if(index!=-1){
			timemask = format.substring(0, index);
			var temp_format = format.substring(index+1);
			index = temp_format.lastIndexOf("_");

			ampm = temp_format.substring(0, index);
			ampmPos = temp_format.substring(index+1);
		}else{
			timemask = format;

		}
		timemask = timemask.toUpperCase();

		//var formatArr=timemask.split(/\W+/);
		var formatArr = timemask.split(/[^H+M+S*]/);


		var timeInfoArray=value.split(/\D+/);

		var timeFrmt = timemask;

		var timeValues = new Array();
		for(var i=0; i<timeInfoArray.length; i++)
		{
			timeValues[formatArr[i]] = timeInfoArray[i];



		}
		var hours,minutes,seconds;
		index = timeFrmt.indexOf("SS");
		if(index==-1){
			index = timeFrmt.indexOf("S");
			if(index!=-1){
				timeFrmt = timeFrmt.substring(0,index)+"$$S$"+timeFrmt.substring(index+1);

				if(timeValues["S"]!=null&&timeValues["S"]!='undefined'){
					if(timeValues["S"].length>1&&timeValues["S"].indexOf("0") ==0){
						timeValues["S"] = timeValues["S"].substring(1);

					}

						seconds = timeValues["S"];
				}
			}

		}else{

			if(timeValues["SS"]!=null&&timeValues["SS"]!='undefined')
				seconds = timeValues["SS"];

			if(seconds!=null&&seconds.indexOf("0")!=0){

				timeValues["SS"] = ((timeValues["SS"]<10&&timeValues["SS"]>0) ? "0":"") + timeValues["SS"];


			}

			timeFrmt = timeFrmt.substring(0,index)+"$$SS$"+timeFrmt.substring(index+2);
		}



		var needAMPM = "false";
		var AM=PM=null;
		if(ampm!='undefined'&&ampm!=null){
			AM = ampm.substring(0, ampm.indexOf("_"));
			PM = ampm.substring(ampm.indexOf("_")+1);
		}
		index = timeFrmt.indexOf("HH");

		if( AM!=""&&AM!=null&&PM!=""&&PM!=null&&index==-1)
			needAMPM = "true";

		var originalHours=0;

		if(index==-1)
		{
			// h instead of hh, need am pm
			index =timeFrmt.indexOf("H");
			if(index!=-1){
				timeFrmt = timeFrmt.substring(0,index)+"$$H$"+timeFrmt.substring(index+1);

				if(timeValues["H"]!=null&&timeValues["H"].length>1&&timeValues["H"].indexOf("0") ==0){
					timeValues["H"] = timeValues["H"].substring(1);

				}
				originalHours = timeValues["H"];
				hours = originalHours;
				if( needAMPM=="true" && timeValues["H"]  > 12 )
				{
				   timeValues["H"] -= 12;
				}

				if(hours=="0"||hours=="00"){
					hours=12;
					timeValues["H"] = "0";
				}
			}

		}else{
			hours = timeValues["HH"];
			if(hours.indexOf("0") !=0){
				timeValues["HH"] = ((timeValues["HH"]<10&&timeValues["HH"]>0) ? "0":"") + timeValues["HH"];

			}

			if(hours=="0"||hours=="00")
				hours=timeValues["HH"] = "00";

			timeFrmt = timeFrmt.substring(0,index)+"$$HH$"+timeFrmt.substring(index+2);
		}

		index = timeFrmt.indexOf("MM");
		if(index==-1)
		{
			// m instead of mm
			index =timeFrmt.indexOf("M");
			if(index!=-1){
				timeFrmt = timeFrmt.substring(0,index)+"$$M$"+timeFrmt.substring(index+1);

				if(timeValues["M"]!=null&&timeValues["M"].length>1&&timeValues["M"].indexOf("0") ==0){
					timeValues["M"] = timeValues["M"].substring(1);

				}

				minutes = timeValues["M"];
			}

		}else{
			minutes = timeValues["MM"];
			if(minutes!=null&&minutes.indexOf("0")!=0)
				timeValues["MM"] = ((timeValues["MM"]<10&&timeValues["MM"]>0) ? "0":"") + timeValues["MM"];
			timeFrmt = timeFrmt.substring(0,index)+"$$MM$"+timeFrmt.substring(index+2);

		}

		hours = (hours!=null&&hours!='undefined')?hours:"00";
		minutes = (minutes!=null&&minutes!='undefined')?minutes:"00";
		seconds = (seconds!=null&&seconds!='undefined')?seconds:"00";

		//alert(hours);
		//alert(minutes);
		//alert(seconds);


		if(!validTime(hours, minutes, seconds)){

			var args = new Array;
			args[0] = value;
			args[1] = format;
			var Msg = NlsFormatMsg(invalid_time, args);
			Log.error("NLSService.getFormattedUsrInputTime", Msg);
			throw new NLSServiceError(Msg);


		}


		var temp = replaceTokens(timeValues, timeFrmt);

		if(temp!="exception"){
			timeStr = temp;
		}else{
			var args = new Array;
			args[0] = value;
			args[1] = format;
			var Msg = NlsFormatMsg(invalid_time, args);
			Log.error("NLSService.getFormattedUsrInputTime", Msg);

			throw new NLSServiceError(Msg);


		}



		//Set up AM or PM format if we have to

		if( needAMPM == "true" && originalHours >= 12)
		{

			if(ampmPos!='undefined'&&ampmPos!=null&&ampmPos == "POST")
		   		timeStr = timeStr + " " + PM ;
		   	else if(ampmPos!='undefined'&&ampmPos!=null&&ampmPos == "PRE")
		   		timeStr = PM + " " + timeStr;
		}
		else if( needAMPM == "true" && originalHours < 12)
		{
			if(ampmPos!='undefined'&&ampmPos!=null&&ampmPos == "POST")
		   		timeStr = timeStr + " " + AM ;
		   	else if(ampmPos!='undefined'&&ampmPos!=null&&ampmPos == "PRE")
		   		timeStr = AM + " " + timeStr;


		}
		else
		{
		   ;//no need to add am or pm
		}

	return timeStr;


	}

/**
* @method public getFormattedUsrInputTime
*   Method to get the formatted time string
* @param Object value
*	Time object like new time(23:25:44)
* @return String
* 	formatted time object string: 23unicode25unicode44unicode [am/pm]
* @exception NLSServiceError
*	if the passes time is not a valid time.
* 	or error when format the time based on the format style.
**/
	this.getFormattedTime= function(value)
	{

//expect input parameter is time object like new time(hh:mm:ss);
            var hours = value.hour;
            var minutes = value.minute;
	        var seconds = value.second;

            var originalHours = hours;
            var index = timeFormat.indexOf("HH");

            var needAMPM = "false";

            if( AM!=""&&AM!=null&&PM!=""&&PM!=null&&index==-1)
                needAMPM = "true";


            if( needAMPM=="true" && hours > 12 )
            {
               hours -= 12;
            }else if(needAMPM=="false"){
				 hours = ((hours  < 10) ? "0":"") + hours;
				 minutes = ((minutes  < 10) ? "0":"") + minutes;
				 seconds = ((seconds  < 10) ? "0":"") + seconds;

			}
            var timeValues = new Object();
            if(index==-1)
            	timeValues["H"] = hours;
            else
            	timeValues["HH"] = hours;

            timeValues["MM"] = minutes;
            timeValues["SS"] = seconds;

			var temp = replaceTokens(timeValues, timeFormat);

			if(temp!="exception"){
				timeStr = temp;
			}else{
				var args = new Array;
				args[0] = value;
				args[1] = timeFormat;
				var Msg = NlsFormatMsg(invalid_time, args);
				Log.error("NLSService.getFormattedUsrInputTime", Msg);
				throw new NLSServiceError(Msg);


			}


            //timeStr = replaceTokens(timeValues, timeFormat);


            //Set up AM or PM format if we have to
            if( needAMPM == "true" && originalHours >= 12)
            {
               timeStr = timeStr + " " + PM ;
            }
            else if( needAMPM == "true" && originalHours < 12)
            {
               timeStr = timeStr + " " + AM ;
            }
            else
            {
               ;//no need to add am or pm
            }

		return timeStr;
	}


/**
* @method public printTime
*   Method to print out the formatted time string
* @param Object value
*	Time object like new time(23:25:44)
* @return String
* 	formatted time object string: 23unicode25unicode44unicode [am/pm]
* @exception NLSServiceError
*	if the passes time is not a valid time.
* 	or error when format the time based on the format style.
**/
	this.printTime = function(value)
	{

		return this.getFormattedTime(value);
	}

	this.printDateTime = function (value)
	{
		//expect input paramter is a datetime object

		return this.printDate(value.dateObj)+" " + this.printTime(value.timeObj);

	}
	this.printDateOnly = function(value)
	{
		//expect input paramter is a datetime object

		return this.printDate(value.dateObj);

	}
	this.printTimeOnly = function(value)
	{
		//expect input paramter is a datetime object

		return this.printTime(value.timeObj);
	}


/**
* @method public getFormattedInteger
*   Method to get the formatted interger
* @param int
*	The integer to be formatted
* @return String
* 	Returns the formatted integer
**/
	this.getFormattedInteger = function (value)
	{

		return formatInteger(value, this.locale);
	}
/**
* @method public getUnformattedInteger
*   Method to get the unformatted integer
* @param String
*	The formatted number to be unformatted
* @return int
* 	Returns the unformatted number
**/
	this.getUnformattedInteger = function (value)
	{

		return getOriginalInt(value, this.locale);

	}

/**
* @method public isValidInteger
*   Method to detect if it a valid integer
* @param int value
*	The integer to be validated
* @return boolean
* 	Returns true if the integer is valid, otherwise false;
**/
	this.isValidInteger = function(value)
	{
		var result = getFormattedInteger(value);
		if(result!="NaN" && result!=null)
			return true;
		else
			return false;

	}
/**
* @method public getFormattedDouble
*   Method to get the formatted double, float etc number.
* @param float/double
*	The number to be formatted
* @return String
* 	Returns the formatted number
**/
	this.getFormattedDouble = function (value)
	{
		//take "USD" as default since we don't need currency des to format double etc.


		return this.formatNumber(value, "USD", this.locale);

	}
/**
* @method public getUnformattedDouble
*   Method to get the unformatted double, float etc number.
* @param String
*	The formatted number to be unformatted
* @return Number
* 	Returns the unformatted number
**/
	this.getUnformattedDouble = function (value)
	{

		return getOriginalNum(value, this.locale);
	}
/**
* @method public getUnformattedDouble
*   Method to detect if it a valid double
* @param float/double
*	The number to be validated
* @return boolean
* 	Returns true if the number is valid, otherwise false;
**/

	this.isValidDouble = function(value)
	{
		var result = getFormattedDouble(value);
		if(result!="NaN" && result!=null)
			return true;
		else
			return false;

	}


	this.printDouble = function (value)
	{
		return this.formatNumber(value, "USD", this.locale);
	}




/**
* @method private removePrefix
*   Method to remove the prefix of currency before unformat currency to pure currency data
* @param String
*	The currency to be handled
* @return String
* 	Returns the pure currency data like"1234.23_USD".
**/
	function removePrefix(valueStr)
	{

		var i=0;
		for(i; i<valueStr.length; i++)
		{

			var temp = parseInt(valueStr.charAt(i));
			if(temp != NaN && temp >= 0 && temp <= 9 ){
				break;
			}else{
				continue;
			}
		}

		return valueStr.substring(i, valueStr.length); //remove prefix

	}


/**
* @method private removeSuffix
*   Method to remove the suffix of currency before unformat currency to pure currency data
* @param String
*	The currency to be handled
* @return String
* 	Returns the pure currency data like"1234.23_USD".
**/
	function removeSuffix(valueStr)
	{
		var i=valueStr.length;
		for(i; i>0; i--){

			var temp = parseInt(valueStr.charAt(i));

			if(temp != NaN && temp >= 0 && temp <= 9){
				break;
			}else{

				continue;
			}
		}
		return valueStr.substring(0, i+1);


	}



/**
* @method private isValidCur
*  Method to validate a given currency value. If no locale is provided,en_US is used by default.
* @param String currency
*	The currency value to be formatted and validated.
* @param String currencyCode
*	The currencyCode used to determine the currency, for example: "USD"
* @param String languageID
*	The locale used to determine the currency.
* @return boolean
* 	Returns false if the currency is not valid. Otherwise true;
**/
	function isValidCur(currency, currencyCode, languageID)
	{
	  var maxFrac;


	  // scientific format is not allowed, for instance, 1e-09
	  var currResult = new String(Trim(currency));
	  if (currResult.indexOf("e") >= 0)
		  return false;

	  var numResult = getUnformattedCur(currency,languageID);

	  if(numResult < 0)
		  return false;

	  currResult = new String(numResult);
	  if (currResult == "NaN" || currResult.indexOf("e") >= 0)
		  return false;

	  if(languageID != null && currencyCode != null && CurrencyFormats[currencyCode]!='undefined')
	  {
		  if(currencyCode=="CAD"){
			  if(languageID=="fr_CA")
			  	maxFrac = CurrencyFormats[currencyCode][1][4];
			  else
			  	maxFrac = CurrencyFormats[currencyCode][0][4];

		  }else{
		  	maxFrac  = CurrencyFormats[currencyCode][4];
		  }

	  }
	  else
	  {
		  maxFrac  = CurrencyFormats["default"][4];
	  }

	   var result = numberOfDecimalPlaces(numResult);


	  if ( result > maxFrac)
	  {

		  return false;
	  }
	  else
	  {

		  return true;
	  }


	}// END isValidCurrency


/**
* @method private formatCurrency
*  Method to format a given currency value. If no locale is provided, en_US is used by default.
* @param String currency
*	The currency value to be formatted
* @param String currencyCode
*	The currencyCode used to determine the currency, for example: "USD"
* @param String languageID
*	The locale used to determine the currency.
* @return String
*  	Returns the formatted currency if successful; string "NaN" if an invalid
* 	currency is provided or null if an unsupported locale value was provided.
**/
	this.formatCurrency = function(currency, currencyCode, languageID)
	{
	  var thouSep;
	  var decSep;
	  var minFrac;
	  var maxFrac;
	  var groupingSize;
	  var secondGroupingSize;

	  if ( (currency == null) || (currency == "") )
		return ("NaN");

	  var currencyString = new String(currency);
	  if ( currencyString.indexOf("-") != -1 )
		return ("NaN");

	  if(currencyCode == "CAD"){
		  // If no locale is specified, use en_US as default
		  if(languageID != null  && CurrencyFormats[currencyCode]!=null&& CurrencyFormats[currencyCode]!='undefined' && LocaleFormats[languageID]!=null&& LocaleFormats[languageID]!='undefined')
		  {

			  thouSep    = LocaleFormats[languageID][0];
			  decSep     = LocaleFormats[languageID][2];
			  if(languageID == "fr_CA"){
				  minFrac  = CurrencyFormats[currencyCode][1][5];
				  maxFrac  = CurrencyFormats[currencyCode][1][4];
			  }else{
				  minFrac  = CurrencyFormats[currencyCode][0][5];
				  maxFrac  = CurrencyFormats[currencyCode][0][4];
			  }
			  groupingSize = LocaleFormats[languageID][3];
			  secondGroupingSize = LocaleFormats[languageID][4];
		  }
		  else
		  {
			thouSep    = LocaleFormats["default"][0];
			decSep     = LocaleFormats["default"][2];
			minFrac  = CurrencyFormats["default"][5];
			maxFrac  = CurrencyFormats["default"][4];
			groupingSize = LocaleFormats["default"][3];
		  }

	  }else{
		  // If no locale is specified, use en_US as default
		  if(languageID != null && currencyCode != null && CurrencyFormats[currencyCode]!=null&& CurrencyFormats[currencyCode]!='undefined' && LocaleFormats[languageID]!=null&& LocaleFormats[languageID]!='undefined')
		  {
			  thouSep    = LocaleFormats[languageID][0];
			  decSep     = LocaleFormats[languageID][2];
			  minFrac  = CurrencyFormats[currencyCode][5];
			  maxFrac  = CurrencyFormats[currencyCode][4];
			  groupingSize = LocaleFormats[languageID][3];
			  secondGroupingSize = LocaleFormats[languageID][4];
		  }
		  else
		  {
			thouSep    = LocaleFormats["default"][0];
			decSep     = LocaleFormats["default"][2];
			minFrac  = CurrencyFormats["default"][5];
			maxFrac  = CurrencyFormats["default"][4];
			groupingSize = LocaleFormats["default"][3];
		  }
	  }

	  if(this.decimalPlace !=null&&!isNaN(this.decimalPlace)){

		  minFrac = this.decimalPlace;
		  maxFrac = this.decimalPlace;

	  }

	  return(formatNum(currency, thouSep, decSep, minFrac, maxFrac,groupingSize, secondGroupingSize));

	}// END formatCurrency

/**
* @method private getUnformattedCur
*  Method to Convert a properly formatted currency value to a primative numeric value.
* If no locale is provided, en_US is used by default.
* @param String currency
*	The currency value to be unformatted
* @param String languageID
*	The locale used to determine the currency.
* @return String
* 	Returns a primative numeric value if successfull; string "NaN" if the
* 	conversion has failed or null if an unsupported locale is provided.
**/
	function getUnformattedCur(currency,  languageID)
	{
	  var thouSep;
	  var decSep;
	  var groupingSize;

	  // If no locale is specified, use en_US as default
	  if(languageID != null &&   LocaleFormats[languageID]!=null&& LocaleFormats[languageID]!='undefined')
	  {
		  thouSep    = LocaleFormats[languageID][0];
		  decSep     = LocaleFormats[languageID][2];
		  groupingSize = LocaleFormats[languageID][3];
	  }
	  else
	  {
		thouSep    = LocaleFormats["default"][0];
		decSep     = LocaleFormats["default"][2];
		groupingSize = LocaleFormats["default"][3];
	  }

	  return(toNum(currency, thouSep, decSep,groupingSize));

	}// END currencyToNumber

/**
* @method private toNum
*  Method to  Convert a number to a primitive numeric value
* @param String num
*	The number to be formatted and validated.
* @param String thouSep
*	The thousand separator to be used in validation and formatting.
* @param String decSep
*	The decimal separator to be used in validation and formatting.
* @param int groupingSize
* 	Grouping size is the number of digits between grouping
*   separators in the integer portion of a number
* @return String
*	A primitive numeric value; string "NaN" if the conversion failed.
**/
	function toNum(num, thouSep, decSep, groupingSize)
	{
	  var currChar;
	  var buffer = "";
	  var number = new String(Trim(num));
	  var thouHit = false;
	  var decHit = false;
	  var groupCount = 0;

	  if( (number == null) || (number == "") )
		return("NaN");

	  for(var i=0; i < number.length; i++)
	  {
		currChar = number.charAt(i);
		groupCount++;

	  if ( (( currChar == ' ')  && ((thouSep.charCodeAt(0) == 160) && thouSep.length == 1)) || (currChar == thouSep)  )
	  {
		  if (decHit == true)
		  {
			  return("NaN");
		  }
		  if (thouHit == true)
		  {
			  if (groupCount-1 != groupingSize)
			  {
				  return("NaN");
			  }
		  }
		  thouHit = true;
		  groupCount = 0;
		  continue;
		}
		else if(currChar == decSep)
		{
			decHit = true;

			if (thouHit == true)
			{
			  if (groupCount-1 != groupingSize)
			  {
				  return("NaN");
			  }
			}
			groupCount = 0;
			buffer += ".";
		}

		// Even if the decimal point is not '.' the javascript function Number() accepts '.'
		// We need the following check to prevent this
		else if ( currChar == '.')
		return("NaN");
		else
		  buffer += currChar;
	  }// END for

	  if (decHit == false && thouHit == true && groupCount != groupingSize)
	  {
		return("NaN");
	  }

	  return(Number(buffer));

	}// END toNum


/**
* @method private formatNum
*  Method to format and validate a given number
* @param String num
*	The number to be formatted and validated.
* @param String thouSep
*	The thousand separator to be used in validation and formatting.
* @param String decSep
*	The decimal separator to be used in validation and formatting.
* @param int minFrac
* 	 The minimum length of the number after the decimal
* @param int maxFrac
*	The maximum length of the number after the decimal
* @param int decPlaces
* 	The number of decimal places for rounding purposes.
* @param int groupingSize
* 	Grouping size is the number of digits between grouping
*   separators in the integer portion of a number
* @return String
*	Returns the formatted number of type String or string "NaN" if an invalid
* 	currency value was provided.
**/
	function formatNum(num, thouSep, decSep, minFrac, maxFrac,groupingSize, secondGroupingSize)
	{

	  var segments;
	  var leftSide;     // Value to the left of the decimal place
	  var rightSide;    // Value to the right of the decimal place
	  var decPos;       // Position of the decimal place in the given number
	  var result = "";  // Formatted number that is returned to the user
	  var buffer = "";  // Buffer used to store temporary info

	  // trim the space strip out the "+" and keep the "-" of the num

	  var tempNum= new String(Trim(num));
	  var number = new Object;
	  if(tempNum.charAt(0)=="+")
		 number = tempNum.substring(1);
	  else if (tempNum.charAt(0)=="-") {
		 number = tempNum.substring(1);
		 result += "-";
	  }
	  else
		 number = tempNum;

	  if( (number == null) || (number == "") )
		return("NaN");

	  segments = number.split(decSep);
	//	segments = number.split(".");
	  // Determine the values on the left and right side of the decimal place
	  if( (decPos = number.indexOf(decSep)) == -1)
	  {
		leftSide  = segments[0];
		rightSide = null;
	  }
	  else if(segments.length <= 2)
	  {
		if(decPos > 0)
		{
		  leftSide  = segments[0];
		  rightSide = segments[1];
		}
		else
		{
		  leftSide  = null;
		  rightSide = segments[1];
		}
	  }
	  else
		return("NaN");


	  // Validate the number on the left side of the decimal place
	  if(leftSide == null)
		buffer = "0";
	  else
	  {
		var value;

		if(thouSep!=""){
			segments = leftSide.split(thouSep);
			value = segments[0];
		}else{
			segments = new Array(leftSide);
			value = leftSide;
		}



		if(isNaN(Number(value)))
		  return("NaN");


		if(segments.length == 1){
		  buffer = value;


		}else
		{
		  if(value.length <= groupingSize)
			buffer = value;
		  else
			return("NaN");


		  for(var j=1; j<segments.length; j++)
		  {
			value = segments[j];

			if( isNaN(Number(value)) || (value.length != groupingSize) )
			  return("NaN");

			buffer += value;

		  }// END for
		}// END else
	  }// END else

	  // Validate the number on the right side of the decimal place
	  if(rightSide != null)
	  {
		if(isNaN(Number(rightSide)))
		  return("NaN");

	// allow less that the minimum fraction digits and just format properly
	//    if (rightSide.length < minFrac)
	//      return ("NaN");

		buffer += "." + rightSide;
	  }

	  // if maxFrac is null do not round the fraction digits
	  if ( maxFrac == null)
		buffer = new String( Number(buffer));
	  else
	  {
		// Format validated number...
		if (buffer == Number(buffer).toString()) {
			buffer = round(Number(buffer), maxFrac);
		}
	  }

	  if(isNaN(buffer))
		return("NaN");

	  segments = buffer.split(".");
	  leftSide = segments[0];


	  var offset;
	  buffer = leftSide.length % groupingSize;

	  if(secondGroupingSize !='undefined' && secondGroupingSize!=null){
		 if(buffer ==0)
		 	result += leftSide;
		 else
		 	//for india 1,23,456.09
		 	var leftSideLen = leftSide.length;
		 	var lastGroupVal= leftSide.substring(leftSideLen-groupingSize, leftSideLen);
		 	leftSide = leftSide.substring(0, leftSideLen-groupingSize);
		 	buffer = leftSide.length%secondGroupingSize;

		  	if(buffer == 0)
				offset = secondGroupingSize;
		  	else
				offset = buffer;

		  	result += leftSide.substring(0, offset);

		  	for(var k=offset; k<leftSide.length; k += secondGroupingSize)
		  	{
				result += thouSep + leftSide.substring(k, k+secondGroupingSize);
		  	}// END for

		  	result += thouSep+lastGroupVal;


	  }else{




		  if(buffer == 0)
			offset = groupingSize;
		  else
			offset = buffer;

		  result += leftSide.substring(0, offset);

		  for(var k=offset; k<leftSide.length; k += groupingSize)
		  {
			result += thouSep + leftSide.substring(k, k+groupingSize);
		  }// END for
	  }

	  if(maxFrac != null)
	  {
			rightSide = segments[1];

		if(maxFrac > 0 && rightSide != null)
			result += decSep + rightSide.substring(0, maxFrac);
	  }
	  else
	  {
		if (segments.length == 2)
			result += decSep + rightSide;
	  }


	  return(result);
	}// END formatNum


/**
* @method private formatNum
*  Method to round off a primitive numeric to the given decimal places
* @param Number num
*	The number to be rounded.
* @param int decPlaces
* 	The number of decimal places for rounding purposes.
* @param int groupingSize
* 	Grouping size is the number of digits between grouping
*   separators in the integer portion of a number
* @return Number
* 	Returns the rounder number as a String object; string "NaN" if "num" is not
* 	a valid number.
**/
	function round(num, decPlaces)
	{
	  var decSep = ".";  // Decimal separator
	  var buff;          // Buffer used to determine rounded number
	  var i;             // Location of the decimal place

	  var descrep;
	  var segments;
	  var leftSide;
	  var rightSide;

	  num = Number(num);

	  if(isNaN(num))
		return("NaN");

	  buffer = "" + Math.round(num * Math.pow(10, decPlaces)) / Math.pow(10,
																	   decPlaces);

	  // Damn!! Math.round does not work correctly; so, we must double check
	  // the formatting...
	  i = buffer.indexOf(decSep);
	  segments = buffer.split(".");

	  if(i == 0)
		buffer = "0" + buffer;

	  if(segments.length == 2)
		descrep = decPlaces - segments[1].length;
	  else
	  {
		descrep = decPlaces;

		if(descrep > 0)
		  buffer += decSep;
	  }

	  for(var j=0; j<descrep; j++)
		buffer += "0";

	  return(buffer);
	}// END round


/**
* @method private numberOfDecimalPlaces
*  	Method to return the number of decimal places after the decimal separator
*  	This is necessary because when you convert a numeric to a string it removes the zeroes
* @param Number num
*	A primitive number
* @return int
* 	Returns the length the the decimal place of the number
**/
	function numberOfDecimalPlaces(num)
	{

	  var number = new String(num);


	  var segments = number.split(".");
	  var rightSide;    // Value to the right of the decimal place
	  var decPos;       // Position of the decimal place in the given number

	  if( (number == null) || (number == "") || isNaN(Number(num)))
	  {
		return("NaN");
	  }

	  // Replace the decimal separator with the currency-specific decimal
	  // separator (there isn't a thousand separator since the number was primitive numeric)
	  if( (decPos = number.indexOf(".")) == -1)
	  {
		return 0;
	  }
	  else if(segments.length <= 2)
	  {
		return segments[1].length;
	  }
	  else
	  {
		return("NaN");
	  }


	}


/**
* @method private validDate
*  	Method to validate date info like YYYY,MM,DD format and validates it if is within
* 	the year range of 1900 to 9999
* @param int inYear
*	year of the date
* @param int inMonth
*	month of the date
* @param int inDay
*	day of the date
* @return boolean
* 	Returns true if date is a valid date, false otherwise
**/
	function validDate(inYear,inMonth,inDay)
	{
		inDay = new String(inDay);
		inMonth = new String(inMonth);
		inYear = new String(inYear);

	   if (inDay.length > 0 && inDay.charAt(0) == "0")
		 {
		  inDay = inDay.substring(1, inDay.length);
		 }

	   if (inMonth.length > 0 && inMonth.charAt(0) == "0")
		 {
		  inMonth = inMonth.substring(1, inMonth.length);
		 }

	   if (inYear.length == 4 &&
		   (inMonth.length == 1 || inMonth.length == 2) &&
		   (inDay.length == 1 || inDay.length == 2))
		{
			var day = parseInt(inDay);
			var month = parseInt(inMonth);
			var year = parseInt(inYear);
			var dayString = day.toString();
			var monthString = month.toString();
			var yearString = year.toString();

			if ((year != NaN && yearString.length == 4 && year >= 1900 && year <= 9999 ) &&
			   (month != NaN && month >= 1 && month <= 12 && (monthString.length == inMonth.length)) && (day != NaN && (inDay.length == dayString.length)))
			{

				var daysMonth = getDaysInMonth(month, year);

				if (day >= 1 && day <= daysMonth)
				{

					return true;
				}
			}
			else
			{

				return false;
			}

		 }
		return false;
	}


/**
* @method private getDaysInMonth
*  	Method to get number of days in month
* @param int month
*	month of the date
* @param int year
*	year of the date
* @return int
* 	Returns number of days in that month of that year
**/
	function getDaysInMonth(month, year)
	{
			var days;

			if (month == 1 || month == 3 || month == 5 || month == 7 || month == 8 || month == 10 || month == 12)
			  days = 31;
			else if (month == 4 || month == 6 || month == 9 || month == 11)
			  days = 30;
			else if (month == 2)
			{
			  if (isLeapYear(year)) {
				days = 29;
			  }
			  else {
				days = 28;
			  }
			}
			return (days);
	}



/**
* @method private isLeapYear
*  	Method to check to see if year is a leap year
* @param int Year
*	year of the date
* @return boolean
* 	Returns true if year is a leap year otherwise false
**/
	function isLeapYear(Year)
	{
			if (((Year % 4) == 0) && ((Year % 100) != 0) || ((Year % 400) == 0)) {
			  return (true);
			}
			else {
			  return (false);
			}
	}




/**
* @method private localizedDate
*  	Method to given a date value, returns the string representation of that date correct for the locale
* @param Object value
*	 the dateValue of date object
* @return String
* 	Returns representation of that date for the locale
**/
	function localizedDate( dateValue)
	{
		var dateValues = new Object();
		dateValues["YYYY"] = dateValue.year;
		dateValues["MM"] = dateValue.month;
		dateValues["DD"] = dateValue.day;

		if(dateFormat==null){

			alert(NlsFormatMsg(invalid_date_format, null));
			return;
		}
		return replaceTokens(dateValues, dateFormat);
	}


/**
* @method private replaceTokens
*  	Method to return the real representation of the formatted date or time by given the format and date/time info
* @param Array tokenValues
*	 Array of date information: tokenValue['YYYY'] etc.
* @param String
*	The date/time format
* @return String
* 	Returns representation of the formatted date/time for the locale
**/

	 function replaceTokens(tokenValues, format)
	 {


		var resultString = "";
		var tokenName;
		var nCursor = 0;



		var tokenStart = format.indexOf( "$$" );

		var tokenEnd;

		while( tokenStart != -1 )
		{

		   tokenEnd = format.indexOf( "$", tokenStart+2 );
		   tokenName = format.substring(tokenStart +2, tokenEnd);


		   if(tokenValues[tokenName]!=null&&tokenValues[tokenName]!='undefined'){
		   		resultString += format.substring(nCursor, tokenStart) + tokenValues[tokenName];
			}else{
				return "exception";
			}

		   nCursor = tokenEnd + 1;
		   tokenStart = format.indexOf( "$$", nCursor);

		}
		resultString += format.substring(nCursor, format.length);

		return  resultString;
	 }





/**
* @method private replaceTokens
*  	Method to validate the time to make sure the time is within
* 	hh:0-23; mm:0-59; ss:0-59
* @param int inHour
*	 hour of the time
* @param int inMinute
*	minute of the time
* @param int inSecond
*	second of the time
* @return boolean
* 	Returns true if date is a valid time, false otherwise
**/

	function validTime(inHour,inMinute,inSecond)
	{

		inHour= new String(inHour);
		inMinute = new String(inMinute);
		inSecond = new String(inSecond);

	   if (inHour.length > 0 && inHour.charAt(0) == "0")
		 {
		  inHour = inHour.substring(1, inHour.length);
		 }

	   if (inMinute.length > 0 && inMinute.charAt(0) == "0")
		 {
		  inMinute = inMinute.substring(1, inMinute.length);
		 }

	   if (inSecond.length > 0 && inSecond.charAt(0) == "0")
		 {
		  inSecond = inSecond.substring(1, inSecond.length);
		 }


	   if ((inHour.length == 1 || inHour.length == 2) &&
		   (inMinute.length == 1 || inMinute.length == 2) &&
		   (inSecond.length == 1 || inSecond.length == 2))
		{
			var hour = parseInt(inHour);
			var minute = parseInt(inMinute);
			var second = parseInt(inSecond);


			if ((hour != NaN && hour >= 0 && hour <= 23 ) &&
			   (minute != NaN && minute >= 0 && minute <= 59) &&
			   (second != NaN && second >= 0 && second <= 59))
			{

				return true;

			}
			else
			{
				return false;
			}

		 }
		return false;
	}


/**
* @method private replaceTokens
*  	Method to unformat the formatted time
* @param String value
*	 the formatted time to be handled
* @return boolean
* 	Returns a time object which is not locale based
**/

	 function unformatTime(value)
	 {
		value = value.toString();
		var timeInfoArray=value.split(/\D+/);

		var i=0;
		var tokenName;
		var valueArray = new Array();

		var tokenStart = timeFormat.indexOf( "$$" );
		var tokenEnd;

		while( tokenStart != -1 )
		{
		   tokenEnd = timeFormat.indexOf( "$", tokenStart+2 );
		   tokenName = timeFormat.substring(tokenStart +2, tokenEnd);
		   valueArray[tokenName] = timeInfoArray[i++];

		   nCursor = tokenEnd + 1;
		   tokenStart = timeFormat.indexOf( "$$", nCursor);


		}

		return valueArray["HH"]+":"+valueArray["MM"]+":"+valueArray["SS"];


	 }


/**
* @method private getOriginalInt
*  	Method to convert a properly formatted int string to a primative numeric value.
* @param String intnum
*	The int string to be convertted
* @param string locale
*	The locale used to convert the number string
* @return int
* 	Returns a primative numeric value or null if an unsupported locale is provided.
**/
	function getOriginalInt(intnum,locale)
	{
	  var thouSep;
	  var decSep;
	  var groupingSize;

	  // If no locale is specified, use en_US as default
	  if(locale != null && LocaleFormats[locale]!=null && LocaleFormats[locale]!='undefined')
	  {
			thouSep = LocaleFormats[locale][1];
			decSep  = LocaleFormats[locale][2];
			groupingSize = LocaleFormats[locale][3];
	  }
	  else
	  {
			thouSep = LocaleFormats["default"][1];
			decSep  = LocaleFormats["default"][2];
			groupingSize = LocaleFormats["default"][3];
	  }


	  return(toIntegerNum(intnum, thouSep, decSep,groupingSize));

	}// END getOriginalInt



/**
* @method private getOriginalNum
*  	Method to convert a properly formatted number string to a primative numeric value.
* @param String num
*	The number string to be convertted
* @param string locale
*	The locale used to convert the number string
* @return Number
* 	Returns a primative numeric value or null if an unsupported locale is provided.
**/
	function getOriginalNum(num,locale)
	{

	  var thouSep;
	  var decSep;
	  var groupingSize;

	  // If no locale is specified, use en_US as default
	  if(locale != null && LocaleFormats[locale]!=null && LocaleFormats[locale]!='undefined')
	  {
	      thouSep = LocaleFormats[locale][1];
	      decSep  = LocaleFormats[locale][2];
	      groupingSize = LocaleFormats[locale][3];
	  }
	  else
	  {
	    thouSep = LocaleFormats["default"][1];
	    decSep  = LocaleFormats["default"][2];
	    groupingSize = LocaleFormats["default"][3];
	  }


	  return(toNum(num, thouSep, decSep,groupingSize));

	}// END getOriginalNum


/**
* @method private formatInteger
*  	Method to get a properly formatted string based on a primative numeric value.
* @param String intNum
*	The number to be formatted and validated.
* @param string locale
*	The locale used to format the number string
* @return String
* 	Returns the formatted number of type String or string "NaN" if an invalid
* 	integer value was provided.
**/
	function formatInteger(intNum, languageID)
	{

	  	var thouSep;
	  	var groupingSize;

	  	// If no locale is specified, use en_US as default
	  	if(languageID != null && LocaleFormats[languageID]!=null && LocaleFormats[languageID]!='undefined')
	  	{
	  	    thouSep = LocaleFormats[languageID][1];
	  	    groupingSize = LocaleFormats[languageID][3];
	  	}
	  	else
	  	{
	  	  thouSep = LocaleFormats["default"][1];
	  	  groupingSize = LocaleFormats["default"][3];
		}
 		return(formatInt(intNum, thouSep,groupingSize));
	}



/**
* @method private formatInt
*  	Method to Format and validate a given integer
* @param int num
*	The number to be formatted and validated.
* @param String thouSep
* 	The thousand separator to be used in validation and formatting.
* @param int groupingSize
* 	Grouping size is the number of digits between grouping
*   separators in the integer portion of a number
* @return String
* 	Returns the formatted number of type String or string "NaN" if an invalid
* 	integer value was provided.
**/
	function formatInt(num, thouSep, groupingSize)
	{
	  var number = new String(Trim(num));
	  var segments;
	  var leftSide;     // Value to the left of the decimal place
	  var result = "";  // Formatted number that is returned to the user
	  var buffer = "";  // Buffer used to store temporary info

	  // Validate the number on the left side of the decimal place
	  if( (number == null) || (number == "") )
		return("NaN");
	  else
	  {
		var value;

		segments = number.split(thouSep);
		value = segments[0];

		if(isNaN(Number(value)))
		  return("NaN");

		if(segments.length == 1)
		  buffer = value;
		else
		{
		  if(value.length <= groupingSize)
			buffer = value;
		  else
			return("NaN");

		  for(var j=1; j<segments.length; j++)
		  {
			value = segments[j];

			if( isNaN(Number(value)) || (value.length != groupingSize) )
			  return("NaN");

			buffer += value;

		  }// END for
		}// END else
	  }// END else

	  for(var i=0; i<buffer.length; i++)
	  {
		var ch = Number(buffer.charAt(i));

		if(isNaN(ch))
		  return("NaN");
	  }

	  var offset;

	  number = buffer;

	  if((number.length % groupingSize) == 0)
		offset = groupingSize;
	  else
		offset = number.length % groupingSize;

	  result = number.substring(0, offset);

	  for(var k=offset; k<number.length; k += groupingSize)
	  {
		result += thouSep + number.substring(k, k+groupingSize);
	  }// END for

	  return(result);
	}// END formatInt


/**
* @method private formatInt
*  	Method to Format and validate a given number
* @param Number num
*	The number to be formatted and validated.
* @param String currencyCode
*	The currencyCode used to determine the currency, for example: "USD"
* @param String languageID
*	The locale used to determine the currency.
* @return String
* 	the formatted number of type String if successfull; Nan if an
* 	invalid currency is provided or null if an unsupported locale value was provided.
**/
	this.formatNumber = function(num, currencyCode, languageID)
	{
	  var thouSep;
	  var decSep;
	  var minFrac;
	  var maxFrac;
	  var groupingSize;
	  var secondGroupingSize;

		if(currencyCode=="CAD"){
			  // If no locale is specified, use en_US as default
			  if(languageID != null && CurrencyFormats[currencyCode]!=null&&CurrencyFormats[currencyCode]!='undefined' && LocaleFormats[languageID]!=null && LocaleFormats[languageID]!='undefined')
			  {
				  thouSep    = LocaleFormats[languageID][1];
				  decSep     = LocaleFormats[languageID][2];
				  if(languageID=="fr_CA"){

					minFrac  = CurrencyFormats[currencyCode][1][5];
					maxFrac  = CurrencyFormats[currencyCode][1][4];
				  }else{

					minFrac  = CurrencyFormats[currencyCode][0][5];
					maxFrac  = CurrencyFormats[currencyCode][0][4];

				  }
				  groupingSize = LocaleFormats[languageID][3];
				  secondGroupingSize = LocaleFormats[languageID][4];
			  }
			  else
			  {
				thouSep    = LocaleFormats["default"][1];
				decSep     = LocaleFormats["default"][2];
				minFrac  = CurrencyFormats["default"][5];
				maxFrac  = CurrencyFormats["default"][4];
				groupingSize = LocaleFormats["default"][3];
			  }


		}else{

			  // If no locale is specified, use en_US as default
			  if(languageID != null && currencyCode != null && CurrencyFormats[currencyCode]!=null&&CurrencyFormats[currencyCode]!='undefined' && LocaleFormats[languageID]!=null && LocaleFormats[languageID]!='undefined')
			  {
				  thouSep    = LocaleFormats[languageID][1];
				  decSep     = LocaleFormats[languageID][2];
				  minFrac  = CurrencyFormats[currencyCode][5];
				  maxFrac  = CurrencyFormats[currencyCode][4];
				  groupingSize = LocaleFormats[languageID][3];
				  secondGroupingSize = LocaleFormats[languageID][4];
			  }
			  else
			  {
				thouSep    = LocaleFormats["default"][1];
				decSep     = LocaleFormats["default"][2];
				minFrac  = CurrencyFormats["default"][5];
				maxFrac  = CurrencyFormats["default"][4];
				groupingSize = LocaleFormats["default"][3];
			  }

	  }

	  if(this.decimalPlace !=null && !isNaN(this.decimalPlace)){
		  minFrac = this.decimalPlace;
		  maxFrac = this.decimalPlace;

	  }

	  return(formatNum(num, thouSep, decSep, minFrac, maxFrac,groupingSize,secondGroupingSize));

	}// END formatNumber



/**
* @method private formatInt
*  	Method to validate and convert an Integer number to a primitive numeric value
* @param int intnum
*	The integer number to be formatted and validated.
* @param String thouSep
* 	The thousand separator to be used in validation and formatting.
* @param String decSep
* 	The decimal separator to be used in validation and formatting.
* @param int groupingSize
* 	Grouping size is the number of digits between grouping
*   separators in the integer portion of a number
* @return int
* 	Returns a primitive numeric value; string "NaN" if the conversion failed.
**/
	function toIntegerNum(intnum, thouSep, decSep, groupingSize)
	{
	  var currChar;
	  var buffer = "";
	  var number = new String(Trim(intnum));
	  var thouHit = false;
	  var decHit = false;
	  var groupCount = 0;

	  if( (number == null) || (number == "") )
		return("NaN");

	  for(var i=0; i < number.length; i++)
	  {
		currChar = number.charAt(i);
		groupCount++;

		if ( (( currChar == ' ')  && ((thouSep.charCodeAt(0) == 160) && thouSep.length == 1)) || (currChar == thouSep)  )
		{

		  if (thouHit == true)
		  {
			  if (groupCount-1 != groupingSize)
			  {
				  return("NaN");
			  }
		  }
		  thouHit = true;
		  groupCount = 0;
		  continue;
		}
		else if(currChar == decSep)
		{
				  return("NaN");
		}
		// Even if the decimal point or thousand separator is not '.' the javascript function Number() accepts '.'
		// We need the following check to prevent this
		else if ( currChar == '.')
		return("NaN");
		else
		  buffer += currChar;
	  }// END for

	  if (thouHit == true && groupCount != groupingSize)
	  {
		return("NaN");
	  }

	  return(Number(buffer));

	}// END toNum


}


/**
 * @class public NLSServiceError
 * This class is used to construct error object when error is catched in NLSService class.
 * @constructor NLSServiceError
 *  The constructor takes the caught error description
 * @param String d
 *	Error description from the thrown out.
**/
function NLSServiceError(d)
{
	this.description = d;
	this.toString = function () { return d; }

}