/* 
 * Copyright 2003,2004 Colin Crist
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */

package hermes.ext.sap;

import hermes.Domain;
import hermes.Hermes;
import hermes.HermesAdmin;
import hermes.HermesException;
import hermes.JNDIConnectionFactory;
import hermes.config.DestinationConfig;
import hermes.ext.HermesAdminSupport;
import hermes.impl.ConfigHelper;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.jms.JMSException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import org.apache.log4j.Logger;

import com.sap.jms.JMSConstants;
import com.sap.jms.client.destination.Queue;
import com.sap.jms.client.destination.Topic;
import com.sap.jmx.remote.JmxConnectionFactory;

/**
 * @author michael.koegel@sap.com
 * @author martin-g
 * @version $Id: $
 * 
 * Plugin to retrieve data from SAP NetWeaver04 WebAS Java JMS Provider
 */
/*
 * Disclaimer: The interfaces used within this code are not intended to be used for application development.
 */
public class SAPJMSAdmin extends HermesAdminSupport implements HermesAdmin {
	
	private static final Logger log = Logger.getLogger(SAPJMSAdmin.class);
	
	private MBeanServerConnection mbsc;
	
	private ObjectName jms_provider;
	
	private String virtualProvider;
	
	/**
	 * 
	 */
	@SuppressWarnings("unchecked")
	public SAPJMSAdmin(Hermes hermes, SAPJMSAdminFactory adminFactory, JNDIConnectionFactory connectionFactory) throws HermesException {
		super(hermes);
		virtualProvider = adminFactory.getVirtualProvider();
		
		try {
			log.error("props: " + connectionFactory._getProperties());
			
			mbsc = JmxConnectionFactory.getMBeanServerConnection(JmxConnectionFactory.PROTOCOL_ENGINE_P4,
					connectionFactory._getProperties());
			// Returns a list of one Object per ClusterNode
			Set<ObjectName> names = (Set<ObjectName>)mbsc.queryNames(new ObjectName(
			"com.sap.default:name=jms_provider,j2eeType=SAP_J2EEServiceRuntimePerNode,*"), null);
			// Pick the Object that represents the ClusterNode which hosts this VirtualProvider
			Iterator<ObjectName> iterator = names.iterator();
			while (iterator.hasNext()) {
				jms_provider = iterator.next();
				if(getData().length > 0){
					// Stop searching as we found the right one.
					if (log.isDebugEnabled()) {
						log.debug("Going to use jms_provider: " + jms_provider.getCanonicalName());
					}
					break;
				}
			}
		} catch (Exception e) {
			throw new HermesException(e);
		}
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see hermes.HermesAdmin#getDepth(javax.jms.Destination)
	 */
	public int getDepth(DestinationConfig dConfig) throws JMSException {
		byte sapJMSDestType = JMSConstants.TYPE_QUEUE;
		String destinationName = dConfig.getName();
		int destType = dConfig.getDomain();
		if (destType == Domain.TOPIC.getId()) {
			sapJMSDestType = JMSConstants.TYPE_TOPIC;
		}
		
		try {
			if (log.isDebugEnabled()) {
				log.debug(new StringBuilder("Going to call getMessagesCount() for vp =[")
						.append(virtualProvider)
						.append("], destination name=[")
						.append(destinationName)
						.append("], and destination type=[")
						.append(sapJMSDestType)
						.append("]").toString());
			}
			Integer result = (Integer) mbsc.invoke(jms_provider, "getMessagesCount", 
					new Object[] { virtualProvider, destinationName, sapJMSDestType },
					new String[] { "java.lang.String", "java.lang.String", "byte" });
			return result.intValue();
		} catch (Exception e) {
			throw new HermesException(e);
		}
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see hermes.HermesAdmin#close()
	 */
	public synchronized void close() throws JMSException {
		if (log.isDebugEnabled()) {
			log.debug("Going to close SAP Hermes plugin: " + this.getClass().getName());
		}
		mbsc = null;
		virtualProvider = null;
		jms_provider = null;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see hermes.HermesAdmin#discoverDestinationConfigs()
	 */
	public Collection<DestinationConfig> discoverDestinationConfigs() throws JMSException {
		final Collection<DestinationConfig> rval = new ArrayList<DestinationConfig>();
		try {
			String[] queues = (String[]) mbsc.getAttribute(new ObjectName(
					"com.sap.default:j2eeType=jms,name=jms_provider/DEFAULT/" + virtualProvider), "QueueNames");
			for (String queue : queues) {
				DestinationConfig config = ConfigHelper.createDestinationConfig(queue, Domain.QUEUE);
				config.setClassName(Queue.class.getName());
				rval.add(config);
			}
			String[] topics = (String[]) mbsc.getAttribute(new ObjectName(
					"com.sap.default:j2eeType=jms,name=jms_provider/DEFAULT/" + virtualProvider), "TopicNames");
			for (String topic : topics) {
				DestinationConfig config = ConfigHelper.createDestinationConfig(topic, Domain.TOPIC);
				config.setClassName(Topic.class.getName());
				rval.add(config);
			}
		} catch (Exception e) {
			throw new HermesException(e);
		}
		return rval;
	}
	
	@SuppressWarnings("unchecked")
	public Map getStatistics(DestinationConfig dConfig) throws JMSException {
		Map<String, String> stats = null;
		
		byte sapJMSDestType = JMSConstants.TYPE_QUEUE;
		int destType = dConfig.getDomain();
		if (destType == Domain.TOPIC.getId()) {
			sapJMSDestType = JMSConstants.TYPE_TOPIC;
		}
		
		String destinationName = dConfig.getName();
		
		try {
			if (log.isDebugEnabled()) {
				log.debug(new StringBuilder("Going to call getHermesStats() for vp =[")
						.append(virtualProvider)
						.append("], destination name=[")
						.append(destinationName)
						.append("], and destination type=[")
						.append(sapJMSDestType)
						.append("]").toString());
			}
			
			stats = (Map<String, String>) mbsc.invoke(jms_provider, "getHermesStats", new Object[] {virtualProvider, destinationName, sapJMSDestType},
					new String[] {"java.lang.String", "java.lang.String", "byte"});
			
		} catch (Exception e) {
			throw new HermesException(e);
		}
		
		if(stats == null){
			throw new HermesException("No Data received for VP " + virtualProvider);
		}
		
		return stats;
	}
	
	private String[][] getData() throws JMSException {
		String[][] result;
		try {
			result = (String[][]) mbsc.invoke(jms_provider, "getData", new String[] { virtualProvider },
					new String[] { "java.lang.String" });
		} catch (Exception e) {
			throw new HermesException(e);
		}
		if(result==null){
			throw new HermesException("No Data received for VP " + virtualProvider);
		}
		return result;
	}
	
}